// Nutrition Consultations - Version 3 (HTML/CSS/JS) - Editorial Style

let currentSlide = 0
let slides = []
let dots = []
let slideInterval
let selectedGoal = ''
let consultations = []

function initPage() {
  const currentPage = window.location.pathname.includes('enroll.html') ? 'enroll' : 'index'
  
  if (currentPage === 'enroll') {
    initEnrollPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadConsultations()
  updateStats()
  
  setInterval(() => {
    loadConsultations()
    updateStats()
  }, 500)
}

function loadConsultations() {
  const saved = localStorage.getItem('nutriConsultations')
  
  if (saved) {
    try {
      consultations = JSON.parse(saved)
    } catch (e) {
      consultations = []
    }
  } else {
    consultations = [
      {
        id: 1,
        plan: 'Сбалансированное питание',
        date: '2025-01-20',
        rating: 5,
        photo: 'images/plan-1.jpg'
      },
      {
        id: 2,
        plan: 'Детокс программа',
        date: '2025-01-10',
        rating: 4,
        photo: 'images/plan-2.jpg'
      },
      {
        id: 3,
        plan: 'План для набора массы',
        date: '2024-12-28',
        rating: 5,
        photo: 'images/plan-3.jpg'
      }
    ]
    localStorage.setItem('nutriConsultations', JSON.stringify(consultations))
  }
  
  renderTimeline()
}

function renderTimeline() {
  const container = document.getElementById('timeline-container')
  if (!container) return
  
  if (consultations.length === 0) {
    container.innerHTML = `
      <div class="empty-editorial">
        <div class="empty-content">
          <h2 class="empty-title-editorial">Начните свой путь</h2>
          <p class="empty-text-editorial">Запишитесь на первую консультацию и получите персональный план питания</p>
          <a href="enroll.html" class="empty-button-editorial">Записаться</a>
        </div>
      </div>
    `
    return
  }
  
  container.innerHTML = consultations.map((consultation, index) => {
    const date = new Date(consultation.date).toLocaleDateString('ru-RU', {
      day: 'numeric',
      month: 'long',
      year: 'numeric'
    })
    
    const isLeft = index % 2 === 0
    
    return `
      <div class="timeline-item">
        <div class="timeline-content ${isLeft ? 'left' : ''}">
          <div class="timeline-image">
            <img src="${consultation.photo}" alt="${consultation.plan}">
            <div class="timeline-date-badge">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                <line x1="16" y1="2" x2="16" y2="6"/>
                <line x1="8" y1="2" x2="8" y2="6"/>
                <line x1="3" y1="10" x2="21" y2="10"/>
              </svg>
              <span>${date}</span>
            </div>
          </div>
          <div class="timeline-text">
            <h3 class="timeline-title">${consultation.plan}</h3>
            <div class="timeline-rating">
              <svg width="18" height="18" viewBox="0 0 24 24" fill="currentColor" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/>
              </svg>
              <span>${consultation.rating}/5</span>
            </div>
            <div class="timeline-status">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/>
                <polyline points="22 4 12 14.01 9 11.01"/>
              </svg>
              <span>Активен</span>
            </div>
          </div>
        </div>
      </div>
    `
  }).join('')
}

function updateStats() {
  const progressValue = document.getElementById('progress-value')
  const statConsultations = document.getElementById('stat-consultations')
  const statRating = document.getElementById('stat-rating')
  const statExcellent = document.getElementById('stat-excellent')
  
  if (progressValue) {
    progressValue.textContent = consultations.length
  }
  
  if (statConsultations) {
    statConsultations.textContent = consultations.length
  }
  
  if (statRating) {
    const avgRating = consultations.length > 0 
      ? (consultations.reduce((sum, c) => sum + (c.rating || 0), 0) / consultations.length).toFixed(1)
      : '0.0'
    statRating.textContent = avgRating
  }
  
  if (statExcellent) {
    statExcellent.textContent = consultations.filter(c => c.rating === 5).length
  }
}

function initEnrollPage() {
  const goalButtons = document.querySelectorAll('.goal-item-editorial')
  const dateGroup = document.getElementById('date-group')
  const dateInput = document.getElementById('consultation-date')
  const submitBtn = document.getElementById('submit-btn')
  const form = document.getElementById('enroll-form')
  const successMessage = document.getElementById('success-message')
  
  // Set min date to today
  const today = new Date()
  const maxDate = new Date(today.getTime() + 180 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]
  
  if (dateInput) {
    dateInput.min = minDateStr
    dateInput.max = maxDateStr
  }
  
  goalButtons.forEach(button => {
    button.addEventListener('click', () => {
      goalButtons.forEach(btn => btn.classList.remove('selected'))
      button.classList.add('selected')
      selectedGoal = button.dataset.goal
      
      if (dateGroup) {
        dateGroup.style.display = 'flex'
      }
      
      if (submitBtn) {
        submitBtn.disabled = false
      }
    })
  })
  
  if (dateInput) {
    dateInput.addEventListener('change', () => {
      if (dateInput.value && selectedGoal) {
        if (submitBtn) {
          submitBtn.disabled = false
        }
      }
    })
  }
  
  if (form) {
    form.addEventListener('submit', (e) => {
      e.preventDefault()
      
      if (!selectedGoal || !dateInput.value) return
      
      const goalNames = {
        'weight-loss': 'Похудение',
        'weight-gain': 'Набор массы',
        'healthy-lifestyle': 'ЗОЖ',
        'allergies': 'Аллергии'
      }
      
      const saved = localStorage.getItem('nutriConsultations')
      const consultations = saved ? JSON.parse(saved) : []
      
      const newConsultation = {
        id: Date.now(),
        plan: `План: ${goalNames[selectedGoal]}`,
        date: dateInput.value,
        rating: null,
        photo: 'images/plan-placeholder.jpg'
      }
      
      consultations.push(newConsultation)
      localStorage.setItem('nutriConsultations', JSON.stringify(consultations))
      
      if (successMessage) {
        successMessage.style.display = 'flex'
      }
      
      setTimeout(() => {
        window.location.href = 'index.html'
      }, 2000)
    })
  }
}

function initSlider() {
  slides = document.querySelectorAll('.slide-editorial')
  dots = document.querySelectorAll('.dot-editorial')
  const prevBtn = document.querySelector('.slider-btn-prev-editorial')
  const nextBtn = document.querySelector('.slider-btn-next-editorial')
  
  if (slides.length === 0) return
  
  function showSlide(index) {
    slides.forEach((slide, i) => {
      slide.classList.toggle('active', i === index)
    })
    
    dots.forEach((dot, i) => {
      dot.classList.toggle('active', i === index)
    })
    
    currentSlide = index
  }
  
  function nextSlide() {
    const next = (currentSlide + 1) % slides.length
    showSlide(next)
  }
  
  function prevSlide() {
    const prev = (currentSlide - 1 + slides.length) % slides.length
    showSlide(prev)
  }
  
  if (prevBtn) {
    prevBtn.addEventListener('click', prevSlide)
  }
  
  if (nextBtn) {
    nextBtn.addEventListener('click', nextSlide)
  }
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => showSlide(index))
  })
  
  slideInterval = setInterval(nextSlide, 3000)
  
  // Pause on hover
  const slider = document.querySelector('.slider-container-editorial')
  if (slider) {
    slider.addEventListener('mouseenter', () => clearInterval(slideInterval))
    slider.addEventListener('mouseleave', () => {
      slideInterval = setInterval(nextSlide, 3000)
    })
  }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', initPage)

